import cameraConverter from 'libraries/engines/camera/converter.js'
import contextDrawer   from 'libraries/engines/rendering/context_drawer.js'
import gradientTools   from 'libraries/physics/gradient_tools.js'
import store           from 'systems/store.js'


let defaultTheme = {
  widthRatio : 12,
  innerRatio : 2.5,
  backColor: 'white',
  waypoint: {
    radius: 25,
    alpha: 0.25,
    background: 'white',
    border: {
      color: 'red',
      width: 5
    },
    color: 'red',
    font: 'bold 20px sans-serif'
  }
}


function drawCheckpoint (ctx, cameraData, zone, index, theme) {
  let toCircle = cameraConverter.worldToScreen(cameraData, zone.isoPos)
  toCircle.r = theme.waypoint.radius

  ctx.globalAlpha = theme.waypoint.alpha
  ctx.lineWidth = theme.waypoint.border.width

  contextDrawer.doCircle(ctx, toCircle)
  contextDrawer.fill(ctx, theme.waypoint.background)
  contextDrawer.stroke(ctx, theme.waypoint.border.color)

  ctx.globalAlpha = 1

  ctx.fillStyle = theme.waypoint.color
  ctx.font = theme.waypoint.font
  ctx.textAlign = 'center'
  ctx.textBaseline = 'middle'
  ctx.fillText(index + 1, toCircle.x, toCircle.y)
}




function getLinkWidth (fromNode, toNode, cameraZ) {
  var link = fromNode.links.find((link) => link.a === toNode || link.b === toNode)
  return link.width * cameraZ
}

function drawLineAndCircle (ctx, from, to, width, color) {
  ctx.beginPath()
  ctx.moveTo(from.x, from.y)
  ctx.lineTo(to.x, to.y)
  ctx.strokeStyle = color
  ctx.lineWidth = width
  ctx.stroke()

  ctx.beginPath()
  ctx.fillStyle = color
  ctx.arc(to.x, to.y, width / 2, 0, Math.PI * 2)
  ctx.fill()
}


function asScreenPositions (cameraData, path) {
  return path.map(node => {
    let pos = cameraConverter.worldToScreen(cameraData, node.isoPos)
    return { x: pos.x, y: pos.y, node }
  })
}



function drawGroup (ctx, path, screenPositions, cameraData, ratio, color) {
  for (let i = 1; i < path.length; i++) {
    let from  = screenPositions[i - 1]
    let to    = screenPositions[i]
    let width = getLinkWidth(from.node, to.node, cameraData.position.z) * ratio

    drawLineAndCircle(ctx, from, to, width, color)
  }
}


function drawPath (ctx, cameraData, path, theme) {
  let screenPositions = asScreenPositions(cameraData, path)

  drawGroup(ctx, path, screenPositions, cameraData, theme.widthRatio, theme.backColor)

  ctx.globalAlpha = 1
  ctx.globalCompositeOperation = 'destination-out'

  drawGroup(ctx, path, screenPositions, cameraData, theme.innerRatio, '#FFFFFF')


  ctx.globalCompositeOperation = 'source-over'
}



function render ({ cameraData, pathFinder, theme }, ctx) {
  ctx.clearRect(0, 0, ctx.canvas.width, ctx.canvas.height)

  let targetTheme = theme.path || defaultTheme

  if (pathFinder.activeMode) {
    pathFinder.pathes.forEach(({ path }) => drawPath(ctx, cameraData, path, targetTheme))
    pathFinder.waypoints.forEach((zone, index) => drawCheckpoint(ctx, cameraData, zone, index, targetTheme))
  }
}


function load (callback) {
  let domCanvas = document.querySelector('#renderers .drawPath')
  domCanvas.style.opacity = 0.7

  callback()
}


export default {
  id: 'drawPath',
  load,
  dependencies: ['path', 'renderers', 'themeData'],
  tags: ['pathFinder', 'draw'],
  sendTo: {
    renderers: {
      zIndex: 100,
      name: 'drawPath',
      run: store.makeDependentTo(['cameraData', 'pathFinder', 'theme'], render)
    }
  }
}