import store from 'systems/store.js'
import math  from 'libraries/physics/math.js'


let getClosestLinkedZone
let moduleImport = { pathLib: (lib) => { getClosestLinkedZone = lib.getClosestLinkedZone }}




function getDistance (zoneA, zoneB) {
  let dx = zoneA.position.x - zoneB.position.x
  let dy = zoneA.position.y - zoneB.position.y
  let dz = zoneA.position.z - zoneB.position.z

  return Math.sqrt(dx * dx + dy * dy + dz * dz)
}


function isLinkValid (link, pathConfig) {
  return (
    link.water  < pathConfig.maxWaterHeight   &&
    link.height > pathConfig.minCeilingHeight
  )
}


function findShortestPath (fromZone, toZone, pathConfig) { // Nota - move to A* with lib ?
  if (fromZone === toZone) {
    return null
  }

  let distances = new Map()
  let previous  = new Map()
  let unvisited = new Set()

  distances.set(fromZone.id, 0)
  unvisited.add(fromZone)

  while (unvisited.size > 0) {
    let current = null
    let minDist = Infinity

    for (let zone of unvisited) {
      let dist = distances.get(zone.id) ?? Infinity
      if (dist < minDist) {
        minDist = dist
        current = zone
      }
    }

    if (!current || minDist === Infinity) break
    if (current === toZone) break

    unvisited.delete(current)

    let currentDist = distances.get(current.id)

    for (let link of current.links) {
      let neighbor = link.a === current ? link.b : link.a

      if (!isLinkValid(link, pathConfig)) continue

      let edgeWeight = getDistance(current, neighbor)
      let newDist = currentDist + edgeWeight
      let neighborDist = distances.get(neighbor.id) ?? Infinity

      if (newDist < neighborDist) {
        distances.set(neighbor.id, newDist)
        previous.set(neighbor.id, current)
        unvisited.add(neighbor)
      }
    }
  }

  if (!previous.has(toZone.id) && fromZone !== toZone) {
    return null
  }

  let path = []
  let current = toZone

  while (current) {
    path.unshift(current)
    current = previous.get(current.id)
  }


  let distance = distances.get(toZone.id) ?? 0
  return path.length > 0 ? { path, distance } : null
}




function ensureLinkedZone (zone, graphData) {
  if (zone.links.length) {
    return zone
  }

  return getClosestLinkedZone(zone.isoPos)
}



function calculatePath (pathFinder, pathConfig, graphData) {
  pathFinder.pathes = []

  if (pathFinder.waypoints.length < 2) {
    return
  }


  for (let i = 0; i < pathFinder.waypoints.length - 1; i++) {
    let fromZone = pathFinder.waypoints[i]
    let toZone   = pathFinder.waypoints[i + 1]

    fromZone = ensureLinkedZone(fromZone, graphData)
    toZone   = ensureLinkedZone(toZone, graphData)


    let segmentPath = findShortestPath(fromZone, toZone, pathConfig)

    if (segmentPath) {
      pathFinder.pathes.push(segmentPath)
    } else {
      pathFinder.pathes.push({ path: [], distance: null })
    }
  }
}



function update () {
  store.doWith(['pathFinder', 'pathConfig', 'graphData'], ({pathFinder, pathConfig, graphData}) => {
    if (!pathFinder.activeMode) {
      return
    }

    calculatePath(pathFinder, pathConfig, graphData)
    store.mutate('pathFinder', (pf) => pf.pathes = pathFinder.pathes)
    store.notifyChange('pathFinder', 'path changed')
  })
}



function mount() {
  store.onNotice('pathFinder', ['waypoints changed'],   update)
  store.onNotice('pathConfig', ['path config changed'], update)
}

export default {
  id: 'pathCalculator',
  tags: ['pathFinder'],
  dependencies: ['path', 'pathLib'],
  mount,
  sendTo: moduleImport
}