import store from 'systems/store.js'
import math  from 'libraries/physics/math.js'

let getClosestLinkedZone

let moduleImport = { pathLib: (lib) => { getClosestLinkedZone = lib.getClosestLinkedZone }}



function normalizeString (str) {
  return str
    .toLowerCase()
    .normalize("NFD")
    .replace(/[\u0300-\u036f]/g, '')
    .replace(/\s+/g, '')
}


function getSearchCard () {
  let card = document.createElement('div')
  card.className = 'search-card'

  card.innerHTML = `
    <input class="search-input" placeholder="Search zone..." />
    <div class="search-results"></div>
  `

  return card
}



function onInfoSelected ({ info }) {
  let zone

  if (!info.zones || !info.zones.length) {
    zone = getClosestLinkedZone(info.isoPos)
  } else {
    zone = info.zones[0]
  }


  if (zone) {
    store.mutate('pathFinder', pf => pf.waypoints.push(zone))
    store.notifyChange('pathFinder', 'waypoints changed')
  }
}


function buildSearchIndex () {
  let searchableZones = []

  store.doWith(['graphData'], ({ graphData }) => {
    graphData.infos.forEach(info => {
      searchableZones.push({
        info,
        label: info.content
      })
    })
  })

  return searchableZones
}


function inject () {
  let container = document.querySelector('#path-panel .waypoint-search')

  let card    = getSearchCard()
  let input   = card.querySelector('.search-input')
  let results = card.querySelector('.search-results')

  input.addEventListener('input', () => {
    results.innerHTML = ''

    let q = normalizeString(input.value)
    if (q) {
      updateResults(q, results, input)
    }
  })


  container.appendChild(card)
}


function updateResults (q, results, input) {
  let searchableInfos = buildSearchIndex()

  searchableInfos
    .filter(e => normalizeString(e.label).includes(q))
    .forEach(e => {
      let row = document.createElement('div')
      row.className = 'search-result'
      row.textContent = e.label
      row.onclick = () => {
        onInfoSelected(e)
        input.value = ''
        input.focus()
        results.innerHTML = ''
      }
      results.appendChild(row)
    })
}




export default {
  id: 'pathSearchUi',
  dependencies: ['pathPanel', 'pathLib'],
  tags: ['pathFinder', 'UI'],
  sendTo: Object.assign(moduleImport, {
    pathPanel: {
      className: 'waypoint-search',
      order: 80,
      inject
    }
  }),
  ressources: {
    css: ['path_finder/ui/path_search']
  }
}